*************************************************************
**** GTAP ELECTRICITY DISAGGREGATION USING SPCE  *************
*************************************************************
* Created by Jeffrey C. Peters, jffptrs@gmail.com *

$ontext
This GAMS program disaggregates the GTAPv9 electricity sector into T&D and generation technologies.
INPUTS: GTAPv9 database, LCOE, and GWh production data
OUTPUTS: A disaggregated GTAP
This GAMS prgram accompanies Peters, 2016. Journal of Global Economic Analysis
$offtext

*************************************************************
***************** GTAP INPUT SETS AND DATA  *****************
*************************************************************

$gdxin worksets.gdx
Set      cost1(*)  complete new costs
         com1(*)  complete new commodities
         user1(*)  complete new users;
$load cost1 com1 user1
$gdxin

*orig.har in SplitCom/work
$gdxin orig.gdx
Set      cost(*)
         src(*)
         user(*)
         reg(*)
         typ(*)
         trades(*)
         com(*);
$load cost src user reg typ trades com
Parameter national(cost,src,user,reg,typ),trade(trades,*,reg,reg)
$load national, trade
$gdxin

Alias(reg,reg1,reg2);
Alias (typ,typ1);
Alias (src,src1);

*worksets.har in SplitCom/work
$gdxin worksets.gdx
Set      splitcom(*)  splitcom       commodity to be split 'ely'
         newcom(com1)  newcom         new commodities
         orguser(user1)  orguser        users not split
         orgcom(com)  orgcom         commodites not split
         orgcost(cost1)
         com0(*)  original commodities
         user0(*)  original users
         cost0(*)  original cost;
$load splitcom newcom orguser orgcom orgcost com0 user0 cost0
$gdxin
Alias(newcom,newcom1,newcom2,newind,newind1);

Parameter national0(*,src,*,reg,typ);
national0(cost,src,user,reg,typ)=national(cost,src,user,reg,typ);

Parameter nationalcond(*,user,reg);
nationalcond(cost,user,reg)=sum(src,sum(typ,national(cost,src,user,reg,typ)));

$gdxin expand1.gdx
Set      mar(com1);
$load mar
Parameter vst(mar,reg)
$load vst
$gdxin

Parameter vst1(com1,reg);
vst1(com1,reg)=0;
vst1(mar,reg)=vst(mar,reg);

*************************************************************
********************** NEW SETS  ****************************
*************************************************************

Set newcost(cost1);
newcost(cost1)=YES-cost0(cost1);

Set gastech(newind) /gasbl,gasp/;
Alias (gastech,gastech1);
Set hydrotech(newind) /hydrobl,hydrop/;
Set oiltech(newind) /oilbl,oilp/;
Alias (oiltech,oiltech1);

Set data /tnd,nuclearbl,coalbl,windbl,otherbl,totoil,solarp,totgas,tothydro,import,export/;
Set bl(newind) /nuclearbl,coalbl,windbl,gasbl,oilbl,hydrobl,otherbl/;
Set pk(newind) /oilp,gasp,solarp,hydrop/;

Set newgen(newind);
Set transdist(newind) /tnd/;
newgen(newind) = YES - user0(newind) - transdist(newind);

*************************************************************
*************** GWH DATA FROM FILE  *************************
*************************************************************

Parameter gwhdata(data,reg) GWh data from IEA Energy Balances;
$CALL GDXXRW Basedata_for_GAMS_2011.xlsx par=gwhdata rng=a1:gr12 Cdim=1 Rdim=1 trace=0 maxdupeerrors=1000
$GDXIN Basedata_for_GAMS_2011.gdx
$LOAD gwhdata
$GDXIN

Set load /base,peak/;
Parameter por(load);
por('base')=0.85;
por('peak')=1-por('base');

*************************************************************
*************** LCOE DATA FROM FILE  *************************
*************************************************************

Set levcost /inv,fuel,ownuse,o_m,efftax/;
Alias (levcost,levcost1);
Set nontaxcost(levcost) /inv,fuel,ownuse,o_m/;
alias(nontaxcost,nontaxcost1);
Set olevcost(levcost) /inv,fuel,o_m,efftax/;
Alias (olevcost,olevcost1);
Set orgptax(orgcost) /ptax/;
Set orgnoptax(orgcost);
orgnoptax(orgcost) = YES - orgptax(orgcost);
Alias(orgnoptax,orgnoptax1);
Set invcost(orgcost) /capital/;
Set fuelcost(orgcost) /gas,gdt,coa,oil,p_c/;
Set nofuelcost(orgcost);
nofuelcost(orgcost)=orgcost(orgcost)-fuelcost(orgcost);
Set o_mcost(orgcost);
o_mcost(orgcost)= YES - invcost(orgcost) - fuelcost(orgcost) - orgptax(orgcost);

Set noptax(cost1);
Set ptax(cost1) /ptax/;
noptax(cost1)=cost1(cost1)-ptax(cost1);
Alias (noptax,noptax1);

$gdxin lcoe.gdx
Parameter lcoe(levcost,newind,reg);
$load lcoe
$gdxin

*********************
** LCOE Processing **
*********************

lcoe('efftax','nuclearbl','usa') = -10;
*Oil BL and P are same tech...
lcoe(levcost,'oilp',reg)=lcoe(levcost,'oilbl',reg);
*Assign nuclear fuel to O&M
lcoe('o_m','nuclearbl',reg)=lcoe('o_m','nuclearbl',reg)+lcoe('fuel','nuclearbl',reg);
lcoe('fuel','nuclearbl',reg)=0;

Parameter fuelcostfactor(newind,reg);
fuelcostfactor("coalbl",reg)$gwhdata("coalbl",reg) = [nationalcond("coa","ely",reg)/gwhdata("coalbl",reg)]/
                                                         [nationalcond("coa","ely","usa")/gwhdata("coalbl","usa")];
fuelcostfactor("gasbl",reg)$gwhdata("totgas",reg) = [[nationalcond("gas","ely",reg)+nationalcond("gdt","ely",reg)]/gwhdata("totgas",reg)]/
                                                         [[nationalcond("gas","ely","usa")+nationalcond("gdt","ely","usa")]/gwhdata("totgas","usa")];
fuelcostfactor("oilbl",reg)$gwhdata("totoil",reg) = [[nationalcond("oil","ely",reg)+nationalcond("p_c","ely",reg)]/gwhdata("totoil",reg)]/
                                                         [[nationalcond("oil","ely","usa")+nationalcond("p_c","ely","usa")]/gwhdata("totoil","usa")];
fuelcostfactor("gasp",reg)$gwhdata("totgas",reg)= fuelcostfactor("gasbl",reg);
fuelcostfactor("oilp",reg)$gwhdata("totgas",reg)= fuelcostfactor("oilbl",reg);

*Indicator data
Parameter lcoei(levcost,newind,reg);
lcoei(levcost,newind,reg)=0;
lcoei(levcost,newind,reg)=1$lcoe(levcost,newind,reg);

lcoe("fuel","coalbl",reg)$(lcoei("fuel","coalbl",reg)=0) = fuelcostfactor("coalbl",reg)*lcoe('fuel',"coalbl","usa");
lcoe('fuel',"gasbl",reg)$(lcoei('fuel',"gasbl",reg)=0) = fuelcostfactor("gasbl",reg)*lcoe('fuel',"gasbl","usa");
lcoe('fuel',"oilbl",reg)$(lcoei('fuel',"oilbl",reg)=0) = fuelcostfactor("oilbl",reg)*lcoe('fuel',"oilbl","usa");
lcoe('fuel',"gasp",reg)$(lcoei('fuel',"gasp",reg)=0) = fuelcostfactor("gasp",reg)*lcoe('fuel',"gasp","usa");
lcoe('fuel',"oilp",reg)$(lcoei('fuel',"oilp",reg)=0) = fuelcostfactor("oilp",reg)*lcoe('fuel',"oilp","usa");

lcoei(levcost,newind,reg)=1$lcoe(levcost,newind,reg);
*average lcoe for regions w/o lcoe data (this is misleading for fuel...)
lcoe(nontaxcost,newind,reg)$(lcoei(nontaxcost,newind,reg)=0)=
(sum(reg1,lcoe(nontaxcost,newind,reg1))/sum(reg1,lcoei(nontaxcost,newind,reg1)))$sum(reg1,lcoei(nontaxcost,newind,reg1));

*Convert to Million$/GWh
lcoe(levcost,newind,reg)=lcoe(levcost,newind,reg)/1000;
*2010USD to 2011USD (region-specific??)  http://data.bls.gov/cgi-bin/cpicalc.pl?cost1=100&year1=2010&year2=2011
lcoe(levcost,newind,reg)=lcoe(levcost,newind,reg)*1.0316;

Parameter lcoetotal(newind,reg);
lcoetotal(newgen,reg) = sum(levcost,lcoe(levcost,newgen,reg));

*****************************************************
********** STAGE 1: GWH OPTIMIZATION ALGORITHM  *****
*****************************************************

*Variables and bounds for equations below
Variables gwh(newind,reg),blpobjval;
gwh.lo(newind,reg)=0;
gwh.fx('tnd',reg) = 0;
gwh.fx('nuclearbl',reg) = gwhdata('nuclearbl',reg);
gwh.fx('coalbl',reg) = gwhdata('coalbl',reg);
gwh.fx('windbl',reg) = gwhdata('windbl',reg);
gwh.fx('solarp',reg) = gwhdata('solarp',reg);
gwh.fx('otherbl',reg) = gwhdata('otherbl',reg);

*****************************************************
****** Here we minimize the total O&M+Fuel cost *****
*****************************************************
* This can be improved by separating fixed and variable O&M cost!

Equation blpobj pseudo-variable cost minimization for baseload;

blpobj ..   sum(reg,
                 sum(bl,gwh(bl,reg)*(lcoe('o_m',bl,reg)+lcoe('fuel',bl,reg)))
                 )
                 =E=blpobjval;

****************************************************
****** Hit baseload target, conservation of GWh ****
****************************************************

* Greater than since we are minimizing baseload costs
Equation blpsplit(reg) baseload peaking split;
blpsplit(reg).. sum(bl,gwh(bl,reg)) =G= por('base')*sum(newgen,gwh(newgen,reg));

Equation gwhoil(data,reg) physical clearance for GWh;
Equation gwhhydro(data,reg) physical clearance for GWh;
Equation gwhgas(data,reg) physical clearance for GWh;

gwhoil('totoil',reg)     ..  sum(oiltech,gwh(oiltech,reg))      =E= gwhdata('totoil',reg);
gwhhydro('tothydro',reg)  ..  sum(hydrotech,gwh(hydrotech,reg)) =E= gwhdata('tothydro',reg);
gwhgas('totgas',reg)    ..  sum(gastech,gwh(gastech,reg))     =E= gwhdata('totgas',reg);

Model blp /blpobj,
         blpsplit,
         gwhoil,
         gwhhydro,
         gwhgas/;

Solve blp using nlp minimizing blpobjval;

Parameter gwhresult(newind,reg);
gwh.l('tnd',reg)=sum(newgen,gwh.l(newgen,reg));
gwhresult(newind,reg) = gwh.l(newind,reg);
Parameter blpsplitresult(reg);
blpsplitresult(reg) = (sum(bl,gwh.l(bl,reg))/sum(newgen,gwh.l(newgen,reg)))$sum(newgen,gwh.l(newgen,reg));

Alias (fuelcost,fuelcost1);
alias (orgcost,orgcost1);
Set gasfuel(fuelcost) /gas,gdt/;
Alias (gasfuel,gasfuel1);
Set oilfuel(fuelcost) /oil,p_c/;
Alias (oilfuel,oilfuel1);

Alias (invcost,invcost1);
Alias (o_mcost,o_mcost1);
Alias (fuelcost,fuelcost1);
Alias (orgptax,orgptax1);

Alias (newgen,newgen1,newgen2);

*******************************************************
* STAGE 2a: Here we minimize deviation from LCOE data *
*******************************************************

Parameter elytotal(levcost,reg);
elytotal('inv',reg) = sum(invcost,nationalcond(invcost,'ely',reg));
elytotal('ownuse',reg) = nationalcond('ely','ely',reg);
*elytotal('ownuse',reg)=sum(typ,national0('ely','dom','ely',reg,typ));
elytotal('o_m',reg) = sum(o_mcost,nationalcond(o_mcost,'ely',reg));
elytotal('fuel',reg) = sum(fuelcost,nationalcond(fuelcost,'ely',reg));
elytotal('efftax',reg) = sum(orgptax,nationalcond(orgptax,'ely',reg));

*****Characterize TnD sector ***************************************************
Parameter xt(cost1,newind,reg),t_tnd(reg), xt_tnd(cost1,reg);

*Allocation to transmission and distribution
*Could be function of population density, centrality of population
t_tnd(reg) = .21;


xt_tnd(orgcost,reg)=nationalcond(orgcost,'cmn',reg);
xt_tnd('cmn',reg) = nationalcond('ely','cmn',reg);
xt_tnd('tnd',reg) = nationalcond('cmn','cmn',reg);
xt_tnd('coa',reg)$gwh.l('CoalBL',reg)=0;
xt_tnd('oil',reg)$sum(oiltech,gwh.l(oiltech,reg))=0;
xt_tnd('gas',reg)$sum(gastech,gwh.l(gastech,reg))=0;
xt_tnd('gdt',reg)$sum(gastech,gwh.l(gastech,reg))=0;

lcoe('inv','tnd',reg)=(sum(invcost,xt_tnd(invcost,reg))/sum(noptax1,xt_tnd(noptax1,reg))
                                 *t_tnd(reg)*sum(noptax1,nationalcond(noptax1,'ely',reg))
                                 /sum(newgen,gwh.l(newgen,reg)))$sum(newgen,gwh.l(newgen,reg));
lcoe('o_m','tnd',reg)=(sum(o_mcost,xt_tnd(o_mcost,reg))/sum(noptax1,xt_tnd(noptax1,reg))
                                 *t_tnd(reg)*sum(noptax1,nationalcond(noptax1,'ely',reg))
                                 /sum(newgen,gwh.l(newgen,reg)))$sum(newgen,gwh.l(newgen,reg));
lcoe('fuel','tnd',reg)=(sum(fuelcost,xt_tnd(fuelcost,reg))/sum(noptax1,xt_tnd(noptax1,reg))
                                 *t_tnd(reg)*sum(noptax1,nationalcond(noptax1,'ely',reg))
                                 /sum(newgen,gwh.l(newgen,reg)))$sum(newgen,gwh.l(newgen,reg));

*Allcoate PTAX evenly across GWh
Parameter remainingtax(reg),allocatetax(newind,reg);
remainingtax(reg) = (nationalcond('ptax','ely',reg)-sum(newgen,lcoe('efftax',newgen,reg)*gwh.l(newgen,reg)));
allocatetax('tnd',reg)$gwh.l('tnd',reg) = t_tnd(reg)*remainingtax(reg)/gwh.l('tnd',reg);
allocatetax(newgen,reg)$gwh.l(newgen,reg) = ((1-t_tnd(reg))*remainingtax(reg)
                                                 /sum(newgen1,gwh.l(newgen1,reg))
                                                 )$sum(newgen1,gwh.l(newgen1,reg));

lcoe('efftax',newind,reg)$(gwh.l(newind,reg)) = allocatetax(newind,reg)+ lcoe('efftax',newind,reg);

lcoe('ownuse','tnd',reg)$sum(newgen,gwh.l(newgen,reg)) = t_tnd(reg)*elytotal('ownuse',reg)/sum(newgen1,gwh.l(newgen1,reg));
lcoe('ownuse',newgen,reg)$gwh.l(newgen,reg) = (1-t_tnd(reg))*elytotal('ownuse',reg)
                                                                 /sum(newgen1,gwh.l(newgen1,reg));

********************************************************************************

Variable lcoe1(levcost,newind,reg),x(orgcost,newind,reg),lcoeobjval;
lcoe1.lo('inv',newind,reg)=1E-20;
lcoe1.lo('o_m',newind,reg)=1E-20;
lcoe1.lo('fuel',newind,reg)=1E-20;
lcoe1.lo('ownuse',newind,reg)=1E-20;
lcoe1.l(levcost,newind,reg)$lcoe(levcost,newind,reg)=lcoe(levcost,newind,reg);
lcoe1.lo('efftax',newind,reg)$(lcoe('efftax',newind,reg) gt 0)=1E-20;
lcoe1.up('efftax',newind,reg)$(lcoe('efftax',newind,reg) lt 0)=-1E-20;
*lcoe1.fx('efftax',newind,reg) = lcoe('efftax',newind,reg);
lcoe1.fx(levcost,newind,reg)$(lcoe(levcost,newind,reg)=0)=0;

* Min/max for LCOE: This prevents huge departures. If it doesn't fit this well, then ad-hoc is fine. See post-sim allocation for solution(reg)=4
Scalar bound;
bound = 10;
lcoe1.lo('inv',newind,reg)=lcoe('inv',newind,reg)/bound;
lcoe1.up('inv',newind,reg)=lcoe('inv',newind,reg)*bound;
lcoe1.lo('o_m',newind,reg)=lcoe('o_m',newind,reg)/bound;
lcoe1.up('o_m',newind,reg)=lcoe('o_m',newind,reg)*bound;
lcoe1.lo('ownuse',newind,reg)=lcoe('ownuse',newind,reg)/bound;
lcoe1.up('ownuse',newind,reg)=lcoe('ownuse',newind,reg)*bound;

Parameter  totcolsum(newind,reg);
totcolsum('tnd',reg) = t_tnd(reg)*(elytotal('o_m',reg)+elytotal('inv',reg)+elytotal('fuel',reg)+elytotal('ownuse',reg));

Set soltyp /1*7/;
Scalar i;
Parameter solution(reg), optsolution(soltyp);
optsolution(soltyp) = 0;

****************
* Objectives   *
****************
*SPCE formulation*
*See: Peters and Hertel, 2016. Matrix balancing with unknown total costs: preserving economic relationships in the electric power sector.
*Economic Systems Research*
Equation minlcoeerrERP(reg);
minlcoeerrERP(reg)$(ord(reg) eq i) ..
                         sum((nontaxcost,newind),
                         ((lcoe1(nontaxcost,newind,reg)*gwh.l(newind,reg)/sum(nontaxcost1,elytotal(nontaxcost1,reg)))*
                         log(
                         (lcoe1(nontaxcost,newind,reg)/sum(nontaxcost1,lcoe1(nontaxcost1,newind,reg)))/
                         (lcoe(nontaxcost,newind,reg)/sum(nontaxcost1,lcoe(nontaxcost1,newind,reg)))
                         ))
                         $(lcoe(nontaxcost,newind,reg) ne 0 and gwh.l(newind,reg))
                         )
                         +
                         sum((nontaxcost,newind),
                         ((lcoe1(nontaxcost,newind,reg)*gwh.l(newind,reg)/sum(nontaxcost1,elytotal(nontaxcost1,reg)))*
                         log(
                         (lcoe1(nontaxcost,newind,reg)*gwh.l(newind,reg)/sum(newind1,lcoe1(nontaxcost,newind1,reg)*gwh.l(newind1,reg)))/
                         (lcoe(nontaxcost,newind,reg)*gwh.l(newind,reg)/sum(newind1,lcoe(nontaxcost,newind1,reg)*gwh.l(newind1,reg)))
                         ))
                         $(lcoe(nontaxcost,newind,reg) ne 0 and gwh.l(newind,reg)and elytotal(nontaxcost,reg) ne 0)
                         )
                         =E=lcoeobjval;
****************
* Constraints  *
****************
* Market clearing constraint (row sum)
Equation mrktclrrow(levcost,reg);
mrktclrrow(levcost,reg)$(ord(reg) eq i) ..
                                 sum(newind,lcoe1(levcost,newind,reg)*gwh.l(newind,reg))+
                                 elytotal(levcost,reg)$(sum(newgen,gwh.l(newgen,reg))eq 0)
                                 =E=
                                 elytotal(levcost,reg);

* Total cost constraint for transmission and distribution
Equation mrktclrtnd(newind,reg);
mrktclrtnd('tnd',reg)$(gwh.l('tnd',reg) and (ord(reg) eq i)) ..
                                  (lcoe1('inv','tnd',reg)
                                  +lcoe1('ownuse','tnd',reg)
                                  +lcoe1('fuel','tnd',reg)
                                  +lcoe1('o_m','tnd',reg))
                                  *gwh.l('tnd',reg)
                                  =E=
                                  totcolsum('tnd',reg);

* Fuel constraints: ensures estimated LCOEs fit with fuel values
Equation coalmin(reg);
coalmin(reg)$(ord(reg) eq i) .. lcoe1('fuel','CoalBL',reg)*gwh.l('CoalBL',reg)
                                 =G=
                                 nationalcond('coa','ely',reg)$gwh.l('CoalBL',reg);
Equation gasmin(reg);
gasmin(reg)$(ord(reg) eq i) .. lcoe1('fuel','GasBL',reg)*gwh.l('GasBL',reg)+lcoe1('fuel','GasP',reg)*gwh.l('GasP',reg)
                                 =G=
                                 (nationalcond('gas','ely',reg)+nationalcond('gdt','ely',reg))$(gwh.l('GasBL',reg) or gwh.l('GasP',reg));
Equation oilmin(reg);
oilmin(reg)$(ord(reg) eq i) .. lcoe1('fuel','OilBL',reg)*gwh.l('OilBL',reg)+lcoe1('fuel','OilP',reg)*gwh.l('OilP',reg)
                                =G= (nationalcond('oil','ely',reg))$(gwh.l('OilBL',reg) or gwh.l('OilP',reg));
Equation oilmax(reg);
oilmax(reg)$(ord(reg) eq i) .. lcoe1('fuel','OilBL',reg)*gwh.l('OilBL',reg)+lcoe1('fuel','OilP',reg)*gwh.l('OilP',reg)
                                =L= sum(oilfuel,nationalcond(oilfuel,'ely',reg))$(gwh.l('OilBL',reg) or gwh.l('OilP',reg));

* Fuel constraints: ensures porportionality
Equation gasrel(reg);
gasrel(reg)$(ord(reg) eq i) .. lcoe1('fuel','GasBL',reg)*lcoe('fuel','GasP',reg)
                                 =E= lcoe1('fuel','GasP',reg)*lcoe('fuel','GasBL',reg);
Equation oilrel(reg);
oilrel(reg)$(ord(reg) eq i) .. lcoe1('fuel','OilBL',reg)*lcoe('fuel','OilP',reg)
                                 =E= lcoe1('fuel','OilP',reg)*lcoe('fuel','OilBL',reg);

Model lcoesplt /
                 minlcoeerrERP,
                 mrktclrrow,
                 mrktclrtnd,
                 coalmin,
                 gasmin,
                 gasrel,
                 oilmin,
                 oilmax
                 oilrel,
                 /;

* Comment out loop, i=ord, and );, comment in i=[desired region]
loop(reg1,
i=ord(reg1);
*i=81;
Solve lcoesplt using nlp minimizing lcoeobjval;
solution(reg1) = lcoesplt.modelstat;
optsolution(soltyp)$(lcoesplt.modelstat eq ord(soltyp)) = optsolution(soltyp)+1;
);

******** ASSIGN LCOE WITH ROW SHARES WHERE DEVIATIONS ARE VERY LARGE ***********
* This occurs primarily due to poor estimates of LCOE where infomration is not known:
* especially fuel costs and the TnD sector when combined with the mrktclrtnd constraint

lcoe1.l('inv',newind,reg)$(solution(reg)=4)= lcoe('inv',newind,reg)*elytotal('inv',reg)/
                                                 sum(newind1,lcoe('inv',newind1,reg)*gwh.l(newind1,reg));

lcoe1.l('o_m',newind,reg)$(solution(reg)=4)= lcoe('o_m',newind,reg)*elytotal('o_m',reg)/
                                                 sum(newind1,lcoe('o_m',newind1,reg)*gwh.l(newind1,reg));

lcoe1.l('ownuse',newind,reg)$(solution(reg)=4)= lcoe('ownuse',newind,reg)*elytotal('ownuse',reg)/
                                                 sum(newind1,lcoe('ownuse',newind1,reg)*gwh.l(newind1,reg));

Parameter fuelallocate(fuelcost,newind,reg);
fuelallocate('coa','coalbl',reg)$(solution(reg)=4 and gwh.l('coalbl',reg)) =
                                                 nationalcond('coa','ely',reg);
fuelallocate(gasfuel,gastech,reg)$(solution(reg)=4 and gwh.l(gastech,reg)) =
                                                 lcoe('fuel',gastech,reg)*gwh.l(gastech,reg)*nationalcond(gasfuel,'ely',reg)/
                                                 sum(gastech1,lcoe('fuel',gastech1,reg)*gwh.l(gastech1,reg));
fuelallocate('oil',oiltech,reg)$(solution(reg)=4 and gwh.l(oiltech,reg)) =
                                                 lcoe('fuel',oiltech,reg)*gwh.l(oiltech,reg)*nationalcond('oil','ely',reg)/
                                                 sum(oiltech1,lcoe('fuel',oiltech1,reg)*gwh.l(oiltech1,reg));
set pctech(newind) /tnd,oilbl,otherbl,oilp/;
alias (pctech,pctech1);
fuelallocate('p_c',pctech,reg)$(solution(reg)=4 and gwh.l(pctech,reg)) =
                                                 lcoe('fuel',pctech,reg)*gwh.l(pctech,reg)*nationalcond('p_c','ely',reg)/
                                                 sum(pctech1,lcoe('fuel',pctech1,reg)*gwh.l(pctech1,reg));
fuelallocate(fuelcost,'tnd',reg)$(solution(reg)=4) =
                                                 nationalcond(fuelcost,'ely',reg) - sum(newgen,fuelallocate(fuelcost,newgen,reg));

lcoe1.l('fuel',newind,reg)$(solution(reg)=4 and gwh.l(newind,reg)) = sum(fuelcost,fuelallocate(fuelcost,newind,reg))/gwh.l(newind,reg);

*Belarus has too large of subsidies. The solution breaks down (negative produciton). Here we assign based on capital costs.
*lcoe1.l('efftax',newind,'blr')= lcoe1.l('inv',newind,'blr')*nationalcond('ptax','ely','blr')/nationalcond('capital','ely','blr');

lcoe1.l('efftax',newind,reg)$(solution(reg)=4 and gwh.l(newind,reg)) = sum(nontaxcost,lcoe1.l(nontaxcost,newind,reg))*
                                                                            elytotal('efftax',reg)/sum(levcost,elytotal(levcost,reg));

**********************************************************************

Parameter        zlcoe(levcost,newind,reg),
                 nlcoe(levcost,newind,reg), nlcoeresult(levcost,*,reg),
                 elcoe(levcost,newind,reg),elcoei(levcost,newind,reg),
                 elcoeshrtech(levcost,newind,reg),elcoeshrcross(levcost,newind,reg);
zlcoe(levcost,newind,reg) = lcoe(levcost,newind,reg)*gwh.l(newind,reg);
nlcoe(levcost,newind,reg)=lcoe1.l(levcost,newind,reg);
nlcoeresult(levcost,newgen,reg)=lcoe1.l(levcost,newgen,reg)*gwh.l(newgen,reg);
nlcoeresult(levcost,'tnd',reg)$sum(newgen,gwh.l(newgen,reg))=lcoe1.l(levcost,'tnd',reg)*sum(newgen,gwh.l(newgen,reg));
nlcoeresult(levcost,'tnd',reg)$(sum(newgen,gwh.l(newgen,reg)) eq 0) =elytotal(levcost,reg);
elcoe(levcost,newind,reg)=(lcoe1.l(levcost,newind,reg)/lcoe(levcost,newind,reg))
                                 $(lcoe(levcost,newind,reg)
                                 and gwh.l(newind,reg));
elcoe(levcost,'tnd',reg)=(lcoe1.l(levcost,'tnd',reg)/lcoe(levcost,'tnd',reg))$lcoe(levcost,'tnd',reg);

elcoeshrtech(levcost,newind,reg) = ((lcoe1.l(levcost,newind,reg)/sum(levcost1,lcoe1.l(levcost1,newind,reg)))/
                                 (lcoe(levcost,newind,reg)/sum(levcost1,lcoe(levcost1,newind,reg))))
                                 $(gwh.l(newind,reg) and lcoe(levcost,newind,reg));

elcoeshrcross(levcost,newind,reg) = ((lcoe1.l(levcost,newind,reg)*gwh.l(newind,reg)/sum(newind1,lcoe1.l(levcost,newind1,reg)*gwh.l(newind1,reg)))/
                                         (lcoe(levcost,newind,reg)*gwh.l(newind,reg)/sum(newind1,lcoe(levcost,newind1,reg)*gwh.l(newind1,reg))))
                                         $(gwh.l(newind,reg) and lcoe(levcost,newind,reg) and sum(newind1,lcoe(levcost,newind1,reg)*gwh.l(newind1,reg))
                                                 and sum(newind1,lcoe1.l(levcost,newind1,reg)*gwh.l(newind1,reg)));

elcoei(levcost,newind,reg)$(elcoe(levcost,newind,reg) gt 5 or elcoe(levcost,newind,reg) lt 0.2) = elcoe(levcost,newind,reg);

***********************************************************
* STAGE 2b: Here we allocate to levelized costs separately *
***********************************************************

Parameter      CWGT(orgcost,src,newind,reg,typ);
Parameter      tCWGTtx(orgcost,newind,reg), tCWGTti(orgcost,newind,reg), t(orgcost,newind,reg);
t(invcost,newind,reg) = 1;

************************************
**************** PTAX **************
************************************
* Simple assignment...
CWGT('ptax','dom',newind,reg,'tax')$sum(newgen,gwh.l(newgen,reg)) = lcoe1.l('efftax',newind,reg)*gwh.l(newind,reg);
CWGT('ptax','dom','tnd',reg,'tax')$(sum(newgen,gwh.l(newgen,reg)) eq 0) =  elytotal('efftax',reg);

************************************
**** Capital (investment costs) ****
************************************
**** We use a 'dual' cross entropy approach with for a the typ differentiation ****

*Power of tax to capital specific to generation typ*
t('capital','SolarP','usa') = .7;

tCWGTti(invcost,newind,reg)$(sum(newind1,gwh.l(newind1,reg)) gt 0 and nationalcond(invcost,'ely',reg) gt 0)
                                = lcoe1.l('inv',newind,reg)*gwh.l(newind,reg)/
                                 (sum(newind1,lcoe1.l('inv',newind1,reg)*gwh.l(newind1,reg)))*
                                 sum((src,typ),national0(invcost,src,'ely',reg,typ));

tCWGTti(invcost,'tnd',reg)$(sum(newind1,gwh.l(newind1,reg)) eq 0 and nationalcond(invcost,'ely',reg) gt 0) =
                                 elytotal('inv',reg);

tCWGTtx(invcost,newind,reg) = tCWGTti(invcost,newind,reg)/t(invcost,newind,reg);

Variable      CWGTtx(orgcost,newind,reg), CWGTti(orgcost,newind,reg), CEinvobjval;

CWGTtx.lo(invcost,newind,reg)=1E-20;
CWGTti.lo(invcost,newind,reg)=1E-20;
CWGTtx.l(invcost,newind,reg)$tCWGTtx(invcost,newind,reg)=tCWGTtx(invcost,newind,reg);
CWGTti.l(invcost,newind,reg)$tCWGTti(invcost,newind,reg)=tCWGTti(invcost,newind,reg);
CWGTtx.fx(invcost,newind,reg)$(tCWGTtx(invcost,newind,reg)=0)=0;
CWGTti.fx(invcost,newind,reg)$(tCWGTti(invcost,newind,reg)=0)=0;

Equation minCEinv(reg);
minCEinv(reg)$(ord(reg) eq i) ..
                                 sum((invcost,newind),
                                 (CWGTtx(invcost,newind,reg)*
                                 log(CWGTtx(invcost,newind,reg)/tCWGTtx(invcost,newind,reg))
                                 )$tCWGTtx(invcost,newind,reg)
                                 )
                                 +
                                 sum((invcost,newind),
                                 (CWGTti(invcost,newind,reg)*
                                 log(CWGTti(invcost,newind,reg)/tCWGTti(invcost,newind,reg))
                                 )$tCWGTti(invcost,newind,reg)
                                 )
                                 =E= CEinvobjval;

Equation mrktclrinvcon1(invcost,reg);
mrktclrinvcon1(invcost,reg)$(ord(reg) eq i) ..
                                 sum(newind,CWGTtx(invcost,newind,reg))
                                 =E=
                                 sum(src,national0(invcost,src,'ely',reg,'bas'));
Equation mrktclrinvcon2(invcost,reg);
mrktclrinvcon2(invcost,reg)$(ord(reg) eq i) ..
                                 sum(newind,CWGTti(invcost,newind,reg))
                                 =E=
                                 sum((src,typ),national0(invcost,src,'ely',reg,typ));

Equation colsuminvcon(newind,reg);
colsuminvcon(newind,reg)$(ord(reg) eq i) ..
                                 sum(invcost,CWGTti(invcost,newind,reg))
                                 =E=
                                 nlcoeresult('inv',newind,reg);
Model CEinv /
            minCEinv,
            mrktclrinvcon1,
            mrktclrinvcon2,
            colsuminvcon
                 /;

Parameter solutioninv(reg), optsolutioninv(soltyp);
optsolutioninv(soltyp) = 0;

loop(reg1,
i=ord(reg1);
*i=29;
Solve CEinv using nlp minimizing CEinvobjval;
solutioninv(reg1) = CEinv.modelstat;
optsolutioninv(soltyp)$(CEinv.modelstat eq ord(soltyp)) = optsolutioninv(soltyp)+1;
);

*Identical domestic/import split as GTAP 'ely' data
CWGT(invcost,src,newind,reg,'bas') = (CWGTtx.l(invcost,newind,reg)*
                                         national0(invcost,src,'ely',reg,'bas')/sum(src1,national0(invcost,src1,'ely',reg,'bas'))
                                         )$sum(src1,national0(invcost,src1,'ely',reg,'bas'));
CWGT(invcost,src,newind,reg,'tax') = ((CWGTti.l(invcost,newind,reg)-CWGTtx.l(invcost,newind,reg))*
                                         national0(invcost,src,'ely',reg,'tax')/sum(src1,national0(invcost,src1,'ely',reg,'tax'))
                                         )$sum(src1,national0(invcost,src1,'ely',reg,'tax'));

************************************
*********** FUEL *******************
************************************

Variable      CWGTf(orgcost,newind,reg), CEfuelobjval;
Set othertech(newind) /tnd, otherbl/;
Alias (othertech,othertech1);
*Initialize
tCWGTti(fuelcost,newind,reg)=0;

tCWGTti('coa',newind,reg)$(gwh.l('CoalBL',reg) = 0 and gwh.l(newind,reg) and lcoe1.l('fuel',newind,reg)) =
                                                         gwh.l(newind,reg)/sum(newind1,gwh.l(newind1,reg))*
                                                         nationalcond('coa','ely',reg);
tCWGTti(gasfuel,newind,reg)$(sum(gastech1,gwh.l(gastech1,reg))=0 and gwh.l(newind,reg) and lcoe1.l('fuel',newind,reg)) =
                                                                 gwh.l(newind,reg)/sum(newind1,gwh.l(newind1,reg))*
                                                                 nationalcond(gasfuel,'ely',reg);
tCWGTti('oil',newind,reg)$(sum(oiltech1,gwh.l(oiltech1,reg))=0 and gwh.l(newind,reg) and lcoe1.l('fuel',newind,reg)) =
                                                                 gwh.l(newind,reg)/sum(newind1,gwh.l(newind1,reg))*
                                                                 nationalcond('oil','ely',reg);
*Detailed fuel for specific power sectors
tCWGTti('coa','CoalBL',reg)$gwh.l('CoalBL',reg) = nationalcond('coa','ely',reg);

tCWGTti(gasfuel,gastech,reg)$sum(gastech1,gwh.l(gastech1,reg)) = nationalcond(gasfuel,'ely',reg)*
                                                                 lcoe1.l('fuel',gastech,reg)*gwh.l(gastech,reg)/
                                                                 sum(gastech1,lcoe1.l('fuel',gastech1,reg)*gwh.l(gastech1,reg));

tCWGTti(oilfuel,oiltech,reg)$sum(oiltech1,gwh.l(oiltech1,reg)) = nationalcond(oilfuel,'ely',reg)*
                                                                 lcoe1.l('fuel',oiltech,reg)*gwh.l(oiltech,reg)/
                                                                 sum(oiltech1,lcoe1.l('fuel',oiltech1,reg)*gwh.l(oiltech1,reg));

parameter constraintaa(reg);
constraintaa(reg)= gasmin.m(reg);

tCWGTti('p_c',newind,reg)$(gwh.l(newind,reg) and lcoe1.l('fuel',newind,reg)) =
                                                 abs(
                                                 nlcoeresult('fuel',newind,reg)-
                                                 sum(fuelcost,tCWGTti(fuelcost,newind,reg))
                                                 );

*If qty of specific fuels is on constraint (i.e., gas and gdt in GasBL and GasP is on constraint)...
tCWGTti(oilfuel,'CoalBL',reg)$(gwh.l('CoalBL',reg) and lcoe1.l('fuel','CoalBL',reg) and coalmin.m(reg) ne 0) = 0;
tCWGTti(oilfuel,gastech,reg)$(gwh.l(gastech,reg) and lcoe1.l('fuel',gastech,reg) and gasmin.m(reg) ne 0) = 0;
tCWGTti(gasfuel,'CoalBL',reg)$(gwh.l('CoalBL',reg) and lcoe1.l('fuel','CoalBL',reg) and coalmin.m(reg) ne 0) = 0;
tCWGTti(gasfuel,oiltech,reg)$(gwh.l(oiltech,reg) and lcoe1.l('fuel',oiltech,reg) and oilmin.m(reg) ne 0) = 0;
tCWGTti('coa',gastech,reg)$(gwh.l(gastech,reg) and lcoe1.l('fuel',gastech,reg) and gasmin.m(reg) ne 0) = 0;
tCWGTti('coa',oiltech,reg)$(gwh.l(oiltech,reg) and lcoe1.l('fuel',oiltech,reg) and oilmin.m(reg) ne 0) = 0;


tCWGTti(fuelcost,'tnd',reg)$(sum(newind1,gwh.l(newind1,reg)) eq 0) =
                                 nationalcond(fuelcost,'ely',reg);

CWGTf.lo(fuelcost,newind,reg)=1E-20;
CWGTf.l(fuelcost,newind,reg)$tCWGTti(fuelcost,newind,reg)=tCWGTti(fuelcost,newind,reg);
CWGTf.fx(fuelcost,newind,reg)$(tCWGTti(fuelcost,newind,reg)=0)=0;


Equation minCEfuel(reg);
minCEfuel(reg)$(ord(reg) eq i) ..
                                 sum((fuelcost,newind),
                                 (CWGTf(fuelcost,newind,reg)*
                                 log(CWGTf(fuelcost,newind,reg)/tCWGTti(fuelcost,newind,reg))
                                 )$tCWGTti(fuelcost,newind,reg)
                                 )
                                 =E= CEfuelobjval;

Equation mrktclrfuelcon(fuelcost,reg);
mrktclrfuelcon(fuelcost,reg)$(ord(reg) eq i) ..
                                 sum(newind,CWGTf(fuelcost,newind,reg))
                                 =E=
                                 sum((src,typ),national0(fuelcost,src,'ely',reg,typ));

Equation colsumfuelcon(newind,reg);
colsumfuelcon(newind,reg)$(ord(reg) eq i) ..
                                 sum(fuelcost,CWGTf(fuelcost,newind,reg))
                                 =E=
                                 nlcoeresult('fuel',newind,reg);

Equation fcoalassgn(reg);
fcoalassgn(reg)$(ord(reg) eq i and gwh.l('CoalBL',reg)) ..
                                 CWGTf('coa','CoalBL',reg)
                                 =E=
                                 nationalcond('coa','ely',reg);
Equation fgasassgn(reg);
fgasassgn(reg)$(ord(reg) eq i and (gwh.l('GasBL',reg) or gwh.l('GasP',reg))) ..
                                 sum((gasfuel,gastech),CWGTf(gasfuel,gastech,reg) )
                                 =E=
                                 sum(gasfuel,nationalcond(gasfuel,'ely',reg));
Equation foilassgn(reg);
foilassgn(reg)$(ord(reg) eq i and (gwh.l('OilBL',reg) or gwh.l('OilP',reg))) ..
                                 sum(oiltech,CWGTf('oil',oiltech,reg))
                                 =E=
                                 nationalcond('oil','ely',reg);
Equation fgasrel(gasfuel,reg);
fgasrel(gasfuel,reg)$(ord(reg) eq i and (gwh.l('GasBL',reg) or gwh.l('GasP',reg))) ..
                                 CWGTf(gasfuel,'GasBL',reg)*lcoe1.l('fuel','GasP',reg)*gwh.l('GasP',reg)
                                 =E=
                                 CWGTf(gasfuel,'GasP',reg)*lcoe1.l('fuel','GasBL',reg)*gwh.l('GasBL',reg);
*Made to be linear...
Equation foilrel(oilfuel,reg);
foilrel(oilfuel,reg)$(ord(reg) eq i and (gwh.l('OilBL',reg) or gwh.l('OilP',reg))) ..
                                 CWGTf(oilfuel,'OilBL',reg)*lcoe1.l('fuel','OilP',reg)*gwh.l('OilP',reg)
                                 =E=
                                 CWGTf(oilfuel,'OilP',reg)*lcoe1.l('fuel','OilBL',reg)*gwh.l('OilBL',reg);

Model CEfuel     /
                 minCEfuel,
                 mrktclrfuelcon,
                 colsumfuelcon
                 fcoalassgn,
                 fgasassgn,
                 foilassgn,
                 fgasrel,
                 foilrel
                 /;

Parameter solutionfuel(reg), optsolutionfuel(soltyp);
optsolutionfuel(soltyp) = 0;

loop(reg1,
i=ord(reg1);
*i=29;
Solve CEfuel using nlp minimizing CEfuelobjval;
solutionfuel(reg1) = CEfuel.modelstat;
optsolutionfuel(soltyp)$(CEfuel.modelstat eq ord(soltyp)) = optsolutionfuel(soltyp)+1;
);

CWGT(fuelcost,src,newind,reg,typ) = (CWGTf.l(fuelcost,newind,reg)*
                                         national0(fuelcost,src,'ely',reg,typ)/nationalcond(fuelcost,'ely',reg)
                                         )$nationalcond(fuelcost,'ely',reg);

************************************
**Operating and Maintenance Costs***
************************************

Variable      CWGTo(orgcost,newind,reg), CEo_mobjval;

*************************************************************
*************** O&M PROBS DATA FROM FILE  *******************
*************************************************************

Set tndgen /TnDp,GENp/;
Parameter TGp(orgcost,tndgen), GENp(orgcost,newind);

* Probability of being TnD or generation
$CALL GDXXRW TnD_GEN_probs.xlsx par=TGp rng=a1:c66 Cdim=1 Rdim=1 trace=0
$GDXIN TnD_GEN_probs.gdx
$LOAD TGp
$GDXIN

* Probability of O&M cost in each generation
$CALL GDXXRW GEN_probs.xlsx par=GENp rng=a1:m66 Cdim=1 Rdim=1 trace=0
$GDXIN GEN_probs.gdx
$LOAD GENp
$GDXIN

*Initial point
TGp(orgcost,tndgen)$(TGp(orgcost,tndgen)=0)=1E-20;
GENp(orgcost,newind)$(GENp(orgcost,newind)=0)=1E-20;
tCWGTti(o_mcost,newind,reg)=1E-20;
tCWGTti(o_mcost,newgen,reg)$(gwh.l(newgen,reg) eq 0)=0;

tCWGTti(o_mcost,'tnd',reg)$sum(newgen1,gwh.l(newgen1,reg))
                                 = nationalcond(o_mcost,'ely',reg)*t_tnd(reg)*TGp(o_mcost,'TnDp')/
                                 (t_tnd(reg)*TGp(o_mcost,'TnDp')
                                 + (1-t_tnd(reg))*TGp(o_mcost,'GENp')*sum(newgen,gwh.l(newgen,reg)*lcoe1.l('o_m',newgen,reg)*GENp(o_mcost,newgen)));
tCWGTti(o_mcost,newgen,reg)$sum(newgen1,gwh.l(newgen1,reg))
                                  =  nationalcond(o_mcost,'ely',reg)*(1-t_tnd(reg))*TGp(o_mcost,'GENp')*gwh.l(newgen,reg)*lcoe1.l('o_m',newgen,reg)*GENp(o_mcost,newgen)/
                                 (t_tnd(reg)*TGp(o_mcost,'TnDp')
                                 + (1-t_tnd(reg))*TGp(o_mcost,'GENp')*sum(newgen1,gwh.l(newgen1,reg)*lcoe1.l('o_m',newgen1,reg)*GENp(o_mcost,newgen1)));

tCWGTti(o_mcost,'tnd',reg)$sum(newgen1,gwh.l(newgen1,reg))
                                 = nationalcond(o_mcost,'ely',reg)*t_tnd(reg)*TGp(o_mcost,'TnDp')/
                                 (t_tnd(reg)*TGp(o_mcost,'TnDp')+(1-t_tnd(reg))*TGp(o_mcost,'GENp'));
tCWGTti(o_mcost,newgen,reg)$sum(newgen1,gwh.l(newgen1,reg))
                                  =  nationalcond(o_mcost,'ely',reg)*(1-t_tnd(reg))*TGp(o_mcost,'GENp')*gwh.l(newgen,reg)*lcoe1.l('o_m',newgen,reg)*GENp(o_mcost,newgen)/
                                         ((t_tnd(reg)*TGp(o_mcost,'TnDp')+(1-t_tnd(reg))*TGp(o_mcost,'GENp'))*
                                         sum(newgen1,gwh.l(newgen1,reg)*lcoe1.l('o_m',newgen1,reg)*GENp(o_mcost,newgen1)));

CWGTo.lo(o_mcost,newind,reg)=1E-20;
CWGTo.l(o_mcost,newind,reg)$tCWGTti(o_mcost,newind,reg)=tCWGTti(o_mcost,newind,reg);
CWGTo.fx(o_mcost,newind,reg)$(tCWGTti(o_mcost,newind,reg)=0)=0;


Equation minCEo_m(reg);
minCEo_m(reg)$(ord(reg) eq i) ..
                                 sum((o_mcost,newind),
                                 (CWGTo(o_mcost,newind,reg)*
                                 log(CWGTo(o_mcost,newind,reg)/tCWGTti(o_mcost,newind,reg))
                                 )$tCWGTti(o_mcost,newind,reg)
                                 )
                                 =E= CEo_mobjval;

Equation mrktclro_mcon(orgcost,reg);
mrktclro_mcon(o_mcost,reg)$(ord(reg) eq i) ..
                                 sum(newind,CWGTo(o_mcost,newind,reg))
                                 =E=
                                 sum((src,typ),national0(o_mcost,src,'ely',reg,typ));

Equation colsumo_mcon(newind,reg);
colsumo_mcon(newind,reg)$(ord(reg) eq i) ..
                                 sum(o_mcost,CWGTo(o_mcost,newind,reg))
                                 =E=
                                 nlcoeresult('o_m',newind,reg);

Model CEo_m     /
                 minCEo_m,
                 mrktclro_mcon,
                 colsumo_mcon
                 /;

Parameter solutiono_m(reg), optsolutiono_m(soltyp);
optsolutiono_m(soltyp) = 0;

loop(reg1,
i=ord(reg1);
*i=5;
Solve CEo_m using nlp minimizing CEo_mobjval;
solutiono_m(reg1) = CEo_m.modelstat;
optsolutiono_m(soltyp)$(CEo_m.modelstat eq ord(soltyp)) = optsolutiono_m(soltyp)+1;
);

CWGT(o_mcost,src,newind,reg,typ) = (CWGTo.l(o_mcost,newind,reg)*
                                         national0(o_mcost,src,'ely',reg,typ)/nationalcond(o_mcost,'ely',reg)
                                         )$nationalcond(o_mcost,'ely',reg);

*************************************************************
*************** THE OTHER WEIGHT MATRICES  ******************
*************************************************************


Parameter TWGT(trades,newcom,reg,reg2);
TWGT(trades,newcom,reg,reg2) = (sum(levcost,nlcoeresult(levcost,newcom,reg))/
                                         sum((levcost,newcom1),nlcoeresult(levcost,newcom1,reg))*
                                         trade(trades,'ely',reg,reg2)
                                         )$sum((levcost,newcom1),nlcoeresult(levcost,newcom1,reg));

Parameter NATIONAL1(*,src,*,reg,typ),TRADE1(trades,*,reg,reg2);
NATIONAL1(orgcost,src,orguser,reg,typ) = national0(orgcost,src,orguser,reg,typ);
NATIONAL1(orgcost,src,newind,reg,typ) = CWGT(orgcost,src,newind,reg,typ);
TRADE1(trades,orgcom,reg,reg2) = trade(trades,orgcom,reg,reg2);
TRADE1(trades,newcom,reg,reg2) = TWGT(trades,newcom,reg,reg2);

*Here we derive rowtotals for original non-split commodites to allocate demands. Demands INCLUDE imported electricity!
Parameter orgrowtotals(newcom,src,reg,typ),RWGT(newcom,src,orguser,reg,typ),XWGT(newcom,src,newind,reg,typ);

orgrowtotals(newcom,'dom',reg,'bas')=  sum(levcost,nlcoeresult(levcost,newcom,reg))-
                                              sum(reg2,trade1('basic',newcom,reg,reg2))-
                                                 vst1(newcom,reg);

orgrowtotals(newcom,'imp',reg,'bas') = sum((trades,reg2),trade1(trades,newcom,reg2,reg));

XWGT(newcom,src,newind,reg,typ)=0;

RWGT(newcom,'dom',orguser,reg,'bas') = orgrowtotals(newcom,'dom',reg,'bas')/sum(newcom1,orgrowtotals(newcom1,'dom',reg,'bas'))
                                         *national0('ely','dom',orguser,reg,'bas');

XWGT(newcom,'dom',newind,reg,'bas') =  orgrowtotals(newcom,'dom',reg,'bas')/sum(newcom1,orgrowtotals(newcom1,'dom',reg,'bas'))
                                         *nlcoeresult('ownuse',newind,reg)/sum(newind1,nlcoeresult('ownuse',newind1,reg))
                                         *national0('ely','dom','ely',reg,'bas');

RWGT(newcom,'imp',orguser,reg,'bas') = orgrowtotals(newcom,'imp',reg,'bas')/sum(newcom1,orgrowtotals(newcom1,'imp',reg,'bas'))
                                         *national0('ely','imp',orguser,reg,'bas');

XWGT(newcom,'imp',newind,reg,'bas') =  orgrowtotals(newcom,'imp',reg,'bas')/sum(newcom1,orgrowtotals(newcom1,'imp',reg,'bas'))
                                         *nlcoeresult('ownuse',newind,reg)/sum(newind1,nlcoeresult('ownuse',newind1,reg))
                                         *national0('ely','imp','ely',reg,'bas');

RWGT(newcom,src,orguser,reg,'tax') =  RWGT(newcom,src,orguser,reg,'bas')*national0('ely',src,orguser,reg,'tax')/national0('ely',src,orguser,reg,'bas');

XWGT(newcom,src,newind,reg,'tax') =  XWGT(newcom,src,newind,reg,'bas')*national0('ely',src,'ely',reg,'tax')/national0('ely',src,'ely',reg,'bas');

NATIONAL1(newcom,src,newind,reg,typ) = XWGT(newcom,src,newind,reg,typ);

NATIONAL1(newcom,src,orguser,reg,typ) = RWGT(newcom,src,orguser,reg,typ);

*************************************************************
******************* CONSISTENCY CHECK  **********************
*************************************************************

Parameters rowcheck(orgcost,src,reg,typ), colcheck(src,orguser,reg,typ), xcheck(src,reg,typ);
rowcheck(orgcost,src,reg,typ) = sum(newind,CWGT(orgcost,src,newind,reg,typ))-national0(orgcost,src,'ely',reg,typ);
colcheck(src,orguser,reg,typ) = sum(newcom,RWGT(newcom,src,orguser,reg,typ))-national0('ely',src,orguser,reg,typ);
xcheck(src,reg,typ) = sum((newcom,newind),XWGT(newcom,src,newind,reg,typ))- national0('ely',src,'ely',reg,typ);

Parameters indcostcheck(newcom,reg),sdimpcheck(newcom,reg);
indcostcheck(newcom,reg)=                     sum((cost1,src,typ),national1(cost1,src,newcom,reg,typ))-
                                              sum(user1,national1(newcom,'dom',user1,reg,'bas'))-
                                              sum(reg2,trade1('basic',newcom,reg,reg2))-
                                              vst1(newcom,reg);
sdimpcheck(newcom,reg) =                      sum((trades,reg2),trade1(trades,newcom,reg2,reg))-
                                              sum(user1,national1(newcom,'imp',user1,reg,'bas'));

Parameters nnegcheck1(cost1,src,user1,reg,typ),nnegcheck2(cost1,src,user1,reg),tnegcheck(trades,newcom,reg,reg2);
nnegcheck1(cost1,src,user1,reg,typ)$(national1(cost1,src,user1,reg,typ) lt 0) = national1(cost1,src,user1,reg,typ);
nnegcheck2(cost1,src,user1,reg)$(sum(typ,national1(cost1,src,user1,reg,typ)) lt 0) = sum(typ,national1(cost1,src,user1,reg,typ));
tnegcheck(trades,newcom,reg,reg2)$(trade1(trades,newcom,reg,reg2) lt 0) = trade1(trades,newcom,reg,reg2);

parameter check1(newind,reg), check2(fuelcost,reg);
check1(newind,reg)=sum(fuelcost1,CWGTf.l(fuelcost1,newind,reg))-nlcoeresult('fuel',newind,reg);
check2(fuelcost,reg)=sum(newind1,CWGTf.l(fuelcost,newind1,reg))-sum((src,typ),national0(fuelcost,src,'ely',reg,typ));

* Creating a V matrix for GTAP-Power validation *
Set in /ownuse, gas, oil, cap, om, coal, tax/;
parameter natl(in,newind,reg);
natl('ownuse',newind,reg) = sum((newcom,src,typ),XWGT(newcom,src,newind,reg,typ));
natl('gas',newind,reg) = sum((src,typ), CWGT('gas',src,newind,reg,typ)) + sum((src,typ), CWGT('gdt',src,newind,reg,typ)) ;
natl('oil',newind,reg) = sum((src,typ), CWGT('oil',src,newind,reg,typ)) + sum((src,typ), CWGT('p_c',src,newind,reg,typ)) ;
natl('coal',newind,reg) = sum((src,typ), CWGT('coa',src,newind,reg,typ)) ;
natl('cap',newind,reg) = sum((src,typ), CWGT('capital',src,newind,reg,typ)) ;
natl('tax',newind,reg) = sum((src,typ), CWGT('ptax',src,newind,reg,typ)) ;
natl('om',newind,reg) =  sum((o_mcost,src,typ), CWGT(o_mcost,src,newind,reg,typ));

natl(in,newind,'xsa')=0;
natl(in,newind,'xsc')=0;
natl(in,newind,'xtw')=0;




*************************************************************
************************** WRITE TO FILES  ******************
*************************************************************

execute_unload 'erp', orgcost,orguser,newcom,newind,cost1,src,user1,reg,typ,trades,com1,com, levcost, gwhdata
                         gwhresult, blpsplitresult, xt,
                         solution, optsolution,fuelallocate, solutioninv,optsolutioninv,solutionfuel,optsolutionfuel,solutiono_m,optsolutiono_m,
                         zlcoe, lcoe, nlcoeresult, nlcoe,elcoe,elcoei, elcoeshrcross, elcoeshrtech, elytotal,
                         nationalcond,national0,totcolsum,trade,
                         tgp,genp,tCWGTtx,tCWGTti,
                         CWGT,RWGT,TWGT,XWGT,
                         check1,check2,
                         orgrowtotals,rowcheck,colcheck,xcheck,indcostcheck, sdimpcheck,
                         nnegcheck1,nnegcheck2,tnegcheck,
                         constraintaa,
                         national1,trade1, vst1,natl,in;

execute 'gdx2har erp.gdx erp.har > erp.log';
