$title  Calibrate a CDE Demand System using GTAP data
$ontext
 Consider the following implicit function that defines the utility u:
 G(z,u) = sum(i,BETA(i)*u**(e(i)*(1-ALPHA(i)))*z(i)**(1-ALPHA(i))) = 1 where z(i) = p0(i)/c0
 p0 = benchmark price index, and c0 is the benchmark expenditure level

 Allen partial elasticities of substitution:
 sigma(i,j) = ALPHA(i)+ALPHA(j)-sum(k,theta(k)*ALPHA(k))-delta(i,j)*ALPHA(i)/theta(i);

 Income elasticities:
 eta(i) = (sum(k,theta(k)*e(k))**(-1))*(e(i)*(1-ALPHA(i))+sum(k,theta(k)*e(k)*ALPHA(k)))
          + (ALPHA(i)-sum(k,theta(k)*ALPHA(k))); 

 The upper and lower bounds for ALPHA, e, and BETA come from the CDE regulatory conditions: 
 substitution coefficient ALPHA(i) in (0,1) for all i (or ALPHA > 1 for all i)
 expansion parameter e(i) >= 0
 scale parameter BETA(i) >= 0
 References: Takeda (2012); Hertel et al. (1997); Hanoch (1975)
 Equations must be declared with static sets, but they can be assigned with dynamic sets. 
YHC: 11/26/2014
$offtext

$if not set ds $set ds g20
$if not set datadir $set datadir .\input\
$if not set wt $set wt 0
$include gtap8data_old

set     info    Information about this calibration /
        ds      "%ds%",
        datadir "%datadir%",
        workdir "%gams.workdir%"
        date    "%system.date%"
        time    "%system.time%" /;

alias(i,j,k);

set rr(r) dynamic subset of r;
rr(r) = no;

parameters
z(i,r)        normalized price
theta(i,r)    value share in final demand
vafm(i,r)     Aggregate final demand,
delta(i,j,r)  diagonal-one off-diagonal-zero
sigma(i,j,r)  Allen partial elasticity of substitution
epsilon_(i,r) targeted own-price elasticity of demand
eta_(i,r)     targeted income elasticity of demand
p0(i,r)       benchmark price index
q0(i,r)       benchmark consumption level
c0(r)         expenditure level
mc0(r)        marginal cost when u is one
weight(i,r)   weight for the square distance
beta(i,r)     scale coefficient

;

$ontext
table  data  expenditure share compensated own-price elasticity income elasticity and price index

		shr	        vt	        eta_	        p0
dvd.	agri	0.081078	-0.172441	0.400000	1.000000
dvd.	manu	0.233856	-0.547482	1.110000	1.000000
dvd.	serv	0.685066	-0.428958	1.130000	1.000000
row.	agri	0.214957	-0.172441	0.800000	1.000000
row.	manu	0.285103	-0.547482	1.030000	1.000000
row.	serv	0.499940	-0.428958	1.050000	1.000000
;
$offtext

vafm(i,r) = vdfm(i,"c",r)*(1+rtfd0(i,"c",r))+vifm(i,"c",r)*(1+rtfi0(i,"c",r));
theta(i,r) = vafm(i,r) / (vom("c",r)*(1-rto("c",r)));
abort$sum(r, round(abs(1-sum(i,theta(i,r))),5)) "Shares do not add up.";

epsilon_(i,r)            = epsilon(i,r);
eta_(i,r)                = eta(i,r); 

$ontext
theta(i,r)               = data(r,i,"shr");
epsilon_(i,r)            = data(r,i,"vt");
eta_(i,r)                = data(r,i,"eta_"); 
$offtext

p0(i,r)                  = 1;
q0(i,r)                  = theta(i,r)/p0(i,r);
c0(r)                    = sum(i,p0(i,r)*q0(i,r));
delta(i,j,r)             = 0;	
delta(i,j,r)$sameas(i,j) = 1;	
weight(i,r)              = theta(i,r)$(%wt% eq 0) + (1/card(j))$(%wt% ne 0);

*      Finish reading data
*      ---------------------------------------------------------

variables
ALPHA(i,r)   substitution coefficient
V(i,r)       own-price elasticity of demand
E(i,r)       expansion coefficient
ETAV(i,r)    income elasticity of demand     
*BETA(i,r)    scale coefficient
OBJONE       objective value for own-price elasticity calibration
OBJTWO       objective value for income elasticity calibration
OBJTHR       objective value for the dummy
OBJFOR       objective value for the dummy
U(r)         utility
;

*      The equation "e_engel" deals with the case where eta from data doesn't satisfly the Engel aggregation

equations
e_v(i,r)        for v     		
e_eta(i,r)    	for ETAV		
e_objone      	for OBJONE	
e_objtwo      	for OBJTWO	
e_objthr      	for OBJTHR	
e_objfor      	for OBJFOR	
e_engel(r)      Engel aggregation
e_etaside(i,r)  ensure eta & eta_ lies on the same side of one 
e_exp(r)        expenditure function     			       
e_dfn(i,r)    	compensated demand       		       
;

*      Step 1: Calibrating to the own-price elasticity of demand     

e_v(i,rr) ..
V(i,rr)$theta(i,rr) =e= theta(i,rr)*(2*ALPHA(i,rr)-sum(k,theta(k,rr)*ALPHA(k,rr)))-ALPHA(i,rr);

e_objone ..
*OBJONE =e= sum((i,rr),weight(i,rr)*(V(i,rr)-epsilon_(i,rr))*(V(i,rr)-epsilon_(i,rr)));
OBJONE =e= -sum((i,rr),V(i,rr)*(log(V(i,rr)/epsilon_(i,rr))-1));

model demandelas / e_v, e_objone /;

loop(r,
rr(r)          = yes;
ALPHA.L(i,rr)  = 0.5;
ALPHA.UP(i,rr) = 0.99999;
ALPHA.LO(i,rr) = 0.00001;
*ALPHA.LO(i,r)  = 1.00001;


V.L(i,rr)      = epsilon_(i,rr);
OBJONE.L       = 0;
solve demandelas using nlp minimizing OBJONE;
sigma(i,j,r)$theta(i,r) = ALPHA.L(i,r)+ALPHA.L(j,r)-sum(k,theta(k,r)*ALPHA.L(k,r))-delta(i,j,r)*ALPHA.L(i,r)/theta(i,r);
rr(r)          = no;
);

*      Step 2: Calibrating the income elasticity of demand

e_eta(i,rr) ..
ETAV(i,rr) =e= (1/sum(k,theta(k,rr)*E(k,rr)))*(E(i,rr)*(1-ALPHA.L(i,rr))+sum(k,theta(k,rr)*E(k,rr)*ALPHA.L(k,rr)))
          + (ALPHA.L(i,rr)-sum(k,theta(k,rr)*ALPHA.L(k,rr))); 

e_objtwo ..
OBJTWO =e= sum((i,rr),weight(i,rr)*(ETAV(i,rr)-eta_(i,rr))*(ETAV(i,rr)-eta_(i,rr)));

e_engel(rr) ..
sum(i,theta(i,rr)*ETAV(i,rr)) =e= 1;

e_etaside(i,rr) ..
(ETAV(i,rr)-1)*(eta_(i,rr)-1) =g= 0;

model incomeelas /e_objtwo, e_engel, e_eta, e_etaside/;

loop(r,
rr(r)          = yes;
E.LO(i,rr)     = 1e-6;
E.L(i,rr)      = 1;
ETAV.L(i,rr)   = eta_(i,r);

OBJTWO.L       = 0; 
solve incomeelas using nlp minimizing OBJTWO;
rr(r)          = no;
);

*      Step 3: Calibrating the scale coefficient BETA holding the utility level equals one

beta(i,r) = (q0(i,r)/(1-ALPHA.L(i,r)))/sum(j,q0(j,r)/(1-ALPHA.L(j,r)));
U.FX(r)   = 1;

parameter  epsilonv00(i,r)  EPSILONV solved by the CDE calibration routine,
           etav00(i,r)      ETAV solved by the CDE calibration routine
           alpha00(i,r)     ALPHA solved by the CDE calibration routine
           e00(i,r)         E solved by the CDE calibration routine
           u00(r)           U solved by the CDE calibration routine
           beta00(i,r)      beta solved by the CDE calibration routine
           mc00(r)          Marginal cost
;

epsilonv00(i,r) = V.L(i,r);
etav00(i,r)     = ETAV.L(i,r);
alpha00(i,r)    = ALPHA.L(i,r);
e00(i,r)        = E.L(i,r);
u00(r)          = U.L(r);
beta00(i,r)     = beta(i,r);


mc00(r) =

 c0(r)*sum(i,beta(i,r)*E.L(i,r)*(1-ALPHA.L(i,r))*(U.L(r)**(E.L(i,r)*(1-ALPHA.L(i,r))-1))*(p0(i,r)/c0(r))**(1-ALPHA.L(i,r)))
      /sum(i,beta(i,r)*(1-ALPHA.L(i,r))*(U.L(r)**(E.L(i,r)*(1-ALPHA.L(i,r))))*(p0(i,r)/c0(r))**(1-ALPHA.L(i,r)));

parameter data model output;
data(i,r,"theta")      = theta(i,r);
data(i,r,"epsilon")    = epsilon_(i,r); 
data(i,r,"epsilonv00") = epsilonv00(i,r);
data(i,r,"eta")        = eta_(i,r);
data(i,r,"etav")       = ETAV.L(i,r);
data(i,r,"alpha")      = ALPHA.L(i,r);
data(i,r,"e")          = E.L(i,r);
data(i,r,"beta")       = beta(i,r);
data("mc",r,"mc")      = mc00(r);
data(i,r,"weight")     = weight(i,r);

execute_unload ".\output\cdecalib_%ds%_lnobj.gdx"; 
