$title  Calibrate a CDE Demand System using GTAP data
$ontext
Maximum Entropy Approach for the CDE demand calibration:
The code is the revision of "cde.gms" provided by the GTAP center.  All algorithms are unchanged.  Some notations 
for variables or parameters are changed to those consistent with GTAP8inGAMS to faciliate the process of reading 
data prepared by GTAP8inGAMS.  
YHC: Jan 2017
$offtext

$if not set ds $set ds g20
$if not set datadir $set datadir .\input\
$if not set wt $set wt 0
$include gtap8data_old

set     info    Information about this calibration /
        ds      "%ds%",
        datadir "%datadir%",
        workdir "%gams.workdir%"
        date    "%system.date%"
        time    "%system.time%" /;

alias(i,j,k);

set rr(r) dynamic subset of r;
rr(r) = no;

parameters
z(i,r)        normalized price
theta(i,r)    value share in final demand
vafm(i,r)     Aggregate final demand,
delta(i,j,r)  diagonal-one off-diagonal-zero
sigma(i,j,r)  Allen partial elasticity of substitution
*epsilon_(i,r) targeted own-price elasticity of demand
*eta_(i,r)     targeted income elasticity of demand
epsilon_(i,r) realized compensated own-price elasticity of demand
p0(i,r)       benchmark price index
q0(i,r)       benchmark consumption level
c0(r)         expenditure level
mc0(r)        marginal cost when u is one
weight(i,r)   weight for the square distance
beta(i,r)     scale coefficient
uncelas(i,r)  targeted uncompensated own-price demand elasticity
incelas(i,r)  targeted income demand elasticity
bound         to avoid zero division
tt            to scale the part of objective function
regind        index for region
result(r)     the real objective value for each r
;

bound  = 0.000001;
tt     = 1000;

vafm(i,r) = vdfm(i,"c",r)*(1+rtfd0(i,"c",r))+vifm(i,"c",r)*(1+rtfi0(i,"c",r));
theta(i,r) = vafm(i,r) / (vom("c",r)*(1-rto("c",r)));
abort$sum(r, round(abs(1-sum(i,theta(i,r))),5)) "Shares do not add up.";

*epsilon_(i,r)            = epsilon(i,r);
uncelas(i,r)             = epsilon(i,r)-eta(i,r)*theta(i,r);
incelas(i,r)             = eta(i,r); 

$ontext
theta(i,r)               = data(r,i,"shr");
epsilon_(i,r)            = data(r,i,"vt");
eta_(i,r)                = data(r,i,"eta_"); 
$offtext

p0(i,r)                  = 1;
q0(i,r)                  = theta(i,r)/p0(i,r);
c0(r)                    = sum(i,p0(i,r)*q0(i,r));
delta(i,j,r)             = 0;	
delta(i,j,r)$sameas(i,j) = 1;	
weight(i,r)              = theta(i,r)$(%wt% eq 0) + (1/card(j))$(%wt% ne 0);

*      Finish reading data
*      ---------------------------------------------------------

variables
ALPHA(i,r)     substitution coefficient
V(i,r)         own-price elasticity of demand
E(i,r)         expansion coefficient
ETAV(i,r)      income elasticity of demand
ALPHAETP(r)    entropy of ALPHA
EETP(r)        entropy of E
ALPHAPNT(r)    penalty for deviations in ALPHA
EPNT(r)        penalty for deviations in E
AHAT(i,r)      deviation of uncompensated own-price demand elasticity
AHAT1(i,r)     deviation of income demand elasticity
UNCELASAC(i,r) actual uncompensated price elasticity
ALPHAMEAN(r)   mean substitution coefficient
OBJ     
;

equations
    objective(r)
    alphaetpeq(r)
    eetpeq(r)
    alphapnteq(r)
    epnteq(r)
    icmels(i,r)
    ahat1eq(i,r)
    epsnml(r)
    uelasaceq(i,r)  actual price elasticity
    ahateq(i,r)     error in elasticity
    alphameaneq(r)
    ;

* Objective function: maximize the entropy relative to the unknown parameters of the cde function
OBJECTIVE(r)$(ord(r) eq regind)..
    OBJ =E= -TT*(EPNT(R) + ALPHAPNT(R)) + EETP(R) + ALPHAETP(R);

* Penalty for errors in the expansion parameter
EPNTEQ(r)$(ord(r) eq regind)..
    EPNT(r) =E= sum(i, theta(i,r)*sqr(AHAT1(i,r)));

* Deviation of income elasticity
AHAT1EQ(i,r)$(ord(r) eq regind)..
    AHAT1(i,r) =E= ETAV(i,r) - incelas(i,r);

* Income elasticity expression found in Hanoch (1975) or Hertel et al (1990)
icmels(i,r)$(ord(r) eq regind)..

ETAV(i,r) =e= (1/sum(j,theta(j,r)*E(j,r)))*(E(i,r)*(1-ALPHA(i,r))+sum(j,theta(j,r)*E(j,r)*ALPHA(j,r)))
          + (ALPHA(i,r)-sum(j,theta(j,r)*ALPHA(j,r))); 


*    ETAV(i,r) =E= E(i,r) + (ALPHA(i,r) - SUM(j,theta(j,r)*ALPHA(j,r))) 
*                         - (ALPHA(i,r)*E(i,r) - SUM(j,theta(j,r)*ALPHA(j,r)*E(j,r)));
                      
* Penalty for errors in the substitution parameter
alphapnteq(r)$(ord(r) eq regind)..
    ALPHAPNT(r) =E= sum(i, theta(i,r)*sqr(AHAT(i,r)));

* Deviation of uncompensated own-price demand elasticity
ahateq(i,r)$(ord(r) eq regind)..
    AHAT(i,r) =E= UNCELASAC(i,r) - uncelas(i,r);
                      
* This last constraint pertains to the uncompensated direct price elasticities
uelasaceq(i,r)$(ord(r) eq regind)..
    UNCELASAC(i,r) =E= -(1-theta(i,r))*ALPHA(i,r) - theta(i,r)*E(i,r)
                       +  theta(i,r)*(ALPHA(i,r)*E(i,r) - SUM(j,theta(j,r)*ALPHA(j,r)*E(j,r)));
                                         
* Cross entropy of the expansion parameter
eetpeq(r)$(ord(r) eq regind)..
    EETP(r) =E= -SUM(i, theta(i,r)*E(i,r)*log(E(i,r)));

* Normalise the expansion parameter
epsnml(r)$(ord(r) eq regind)..
    SUM(i, theta(i,r)*E(i,r)) =E= 1;

* Cross entropy of the substitution parameter
alphaetpeq(r)$(ord(r) eq regind)..
    ALPHAETP(r) =E= -SUM(i, theta(i,r)*(ALPHA(i,r)*log(ALPHA(i,r)/ALPHAMEAN(r))
                                           +(1-ALPHA(i,r))*log((1-ALPHA(i,r))/(1-ALPHAMEAN(r)))));
                  
* Mean substitution parameter
alphameaneq(r)$(ord(r) eq regind)..
    ALPHAMEAN(r) =E= sum(i,theta(i,r)*ALPHA(i,r));

* Variable bounds
$ontext
ALPHA.LO(i,r)    = bound;
ALPHA.L(i,r)     = 0.5;
ALPHA.UP(i,r)    = 1.0 - bound;
ALPHAMEAN.L(r)   = 0.5;
E.LO(i,r)        = bound;
E.L(i,r)         = 1.0;
$offtext

ALPHA.LO(i,r)    = bound;
ALPHA.L(i,r)     = 0.5;
ALPHA.UP(i,r)    = 1.0 - bound;
ALPHAMEAN.L(r)   = 0.5;
E.LO(i,r)        = bound;
E.L(i,r)         = 1.0;

UNCELASAC.L(i,r) = uncelas(i,r);
ETAV.L(i,r)      = incelas(i,r);


alias (r, rreg);
model cdent /all/;

loop (rreg,
      regind=ord(rreg);
      solve cdent using nlp maximizing obj;
      display obj.l;
      result(rreg) = obj.l + tt*(sum(i,theta(i,rreg)*power(ahat.l(i,rreg),2))      
                                +sum(i,theta(i,rreg)*power(ahat1.l(i,rreg),2)));
      epsilon_(i,r) = UNCELASAC.L(i,r)+ETAV.L(i,r)*theta(i,r);
     );

execute_unload ".\output\cdeetp_%ds%.gdx"; 
