# Load packages
using GlobalTradeAnalysisProjectModelV7, HeaderArrayFile, NamedArrays

# Read the data
parameters = HeaderArrayFile.readHar("./../Data/gsdfpar.har")
data = HeaderArrayFile.readHar("./../Data/gsdfdat.har")
sets = HeaderArrayFile.readHar("./../Data/gsdfset.har")

# Aggregate the data
## Prepare the mapping vector for regions
regMap = NamedArray(sets["reg"], sets["reg"])
regMap[1:3] .= "oceania"
regMap[4:10] .= "easia"
regMap[11:20] .= "seasia"
regMap[21:27] .= "sasia"
regMap[28:31] .= "namerica"
regMap[32:42] .= "samerica"
regMap[43:55] .= "carib"
regMap[56:82] .= "eu"
regMap[83:98] .= "oeurope"
regMap[99:116] .= "wasia"
regMap[117:121] .= "mena"
regMap[122:133] .= "wafrica"
regMap[134:140] .= "scafrica"
regMap[141:154] .= "eafrica"
regMap[155:160] .= "safrica"

## Prepare the mapping vector for commodities
comMap = NamedArray(sets["comm"], sets["comm"])
comMap[1:8] .= "crops"
comMap[9:12] .= "animals"
comMap[13:18] .= "extract"
comMap[19:26] .= "food"
comMap[27:45] .= "manuf"
comMap[46:65] .= "svces"

## Prepare the mapping vector for endowments
endMap = NamedArray(sets["endw"], sets["endw"])
endMap[:] .= "other"
endMap[1:1] .= "land"
endMap[[2, 5]] .= "sklabor"
endMap[[3, 4, 6]] .= "unsklabor"
endMap["capital"] = "capital"


# Do the aggregation
(; hData, hParameters, hSets) = aggregate_data_legacy(hData=data, hParameters=parameters, hSets=sets, comMap=comMap, regMap=regMap, endMap=endMap)

# We set rordelta to 0 in this example
hParameters["rdlt"] .= 0

# Produce initial uncalibrated model using the GTAP data
mc = generate_initial_model(hSets=hSets, hData=hData, hParameters=hParameters)

# Keep the start data for calibration---the value flows are the correct ones
start_data = deepcopy(mc.data)

# Get the required inputs for calibration by providing the target values in start_data
(;fixed_calibration, data_calibration)=generate_calibration_inputs(mc, start_data)

# Keep the default closure (fixed) for later
fixed_default = deepcopy(mc.fixed)

# Load the calibration data and closure 
mc.data = data_calibration
mc.fixed = fixed_calibration

# Run the model to calibrate
run_model!(mc)

# Save the calibrated data---this is the starting point for all simulation
calibrated_data = deepcopy(mc.data)

# Simplify the model now that calibration equations are not needed
rebuild_model!(mc)

# Let's change the closure to the default (simulation) closure
mc.fixed = deepcopy(fixed_default)

# Start with the calibrated data
mc.data = deepcopy(calibrated_data)

# Double productivity of services in some African regions
mc.data["γ_qintva"][["svces"],["wafrica",  "scafrica", "eafrica", "safrica"]] .= mc.data["γ_qintva"][["svces"],["wafrica",  "scafrica", "eafrica", "safrica"]] * 2

# Run the model
run_model!(mc)

# Calculate EV
ev = calculate_expenditure(sets = mc.sets, data0=calibrated_data, data1=mc.data, parameters=mc.parameters)  .- calibrated_data["y"]

# Because quantity GDP index in GTAP is an average growth of GDP valued at original and post-simulation prices,
# we need to do the same here
# Calculate new GDP at old prices
qgdp1 = calculate_gdp(sets =mc.sets, data0=calibrated_data, data1=mc.data)
# Calculate old GDP at old prices
qgdp0 = calculate_gdp(sets =mc.sets, data0=calibrated_data, data1=calibrated_data)
# Calculate change in GDP
changeqgdp0 = (qgdp1./qgdp0 .- 1) .* 100

# Now calculate new GDP at new prices
qgdp1 = calculate_gdp(sets =mc.sets, data0=mc.data, data1=mc.data)
# Calculate old GDP at new prices
qgdp0 = calculate_gdp(sets =mc.sets, data0=mc.data, data1=calibrated_data)
# Calculate change in this GDP
changeqgdp1 = (qgdp1./qgdp0 .- 1) .* 100
# Change in the GDP quantity index is an average of the two
changeqgdp = 0.5 .* (changeqgdp0 .+ changeqgdp1)

# Report some results in CSV files
using Tables, CSV
# Change in domestic prices
CSV.write("technicalchange_example_ppa.csv", Tables.table(hcat(names(calibrated_data["ppa"])[1], (mc.data["ppa"][:, :] ./ calibrated_data["ppa"][:, :] .- 1) .* 100), header = ["commodity";names(calibrated_data["ppa"])[2]]))

# EV
CSV.write("technicalchange_example_ev.csv", Tables.table(hcat(names(ev)[1], ev), header = ["region","ev"]))

# GDP
CSV.write("technicalchange_example_qgdp.csv", Tables.table(hcat(names(changeqgdp)[1], changeqgdp), header = ["region","qgdp"]))
